/**
 * \file: KeyboardHelper.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Test
 *
 * \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2013-2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <linux/input.h>
#include <sys/prctl.h>
#include <algorithm>
#include <adit_logging.h>
#include "KeyboardHelper.h"

#include <sys/prctl.h>


LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

KeyboardHelper::KeyboardHelper()
{
    thread = 0;
    running = false;
    fd = 0;
    isDetection = false;
    receiver = nullptr;
}

KeyboardHelper::~KeyboardHelper()
{
    Stop();
}

bool KeyboardHelper::Start(IKeyboardReceiver* inReceiver)
{
    receiver = inReceiver;

    running = true;
    int rc = pthread_create(&thread, nullptr, &KeyboardHelper::process, (void*)this);
    if (rc > 0)
    {
        LOG_ERROR((demo, "Keyboard helper thread creation failed with %d", rc));
        return false;
    }
    else
    {
        LOGD_DEBUG((demo, "Keyboard helper started"));
    }

    return true;
}

void KeyboardHelper::Stop()
{
    running = false;
    if (thread != 0)
    {
        pthread_join(thread, nullptr);
        thread = 0;
        LOGD_VERBOSE((demo, "Keyboard helper stopped"));
    }
}

void* KeyboardHelper::process(void* inData)
{
    if (inData == nullptr)
        return nullptr;

    auto me = static_cast<KeyboardHelper*>(inData);

    fd_set fds;
    std::string inputEventPath;
    std::string keyboardName;

    // set thread name
    prctl(PR_SET_NAME, "KeyboardHelper", 0, 0, 0);

    if (!me->selectKeyboard(inputEventPath, keyboardName))
    {
        LOG_WARN((demo, "Selecttion keybord fails. USB Keyboard is not exist."));
        return nullptr; // error logged
    }
    else
    {
        me->isDetection = true;
        me->receiver->display();
    }

    me->fd = open(inputEventPath.c_str(), O_RDONLY);
    if (me->fd == -1)
    {
        LOG_ERROR((demo, "%s is not a valid path", inputEventPath.c_str()));
        return nullptr;
    }
    else
    {
        LOG_INFO((demo, "Keyboard helper listens on %s", inputEventPath.c_str()));
    }

    while (me->running)
    {
        struct timeval timeout;
        timeout.tv_sec =  0;
        timeout.tv_usec = 500000;

        FD_ZERO(&fds);
        FD_SET(me->fd, &fds);
        int ret = select(me->fd + 1, &fds, NULL, NULL, &timeout);
        if (ret > 0 && FD_ISSET(me->fd, &fds))
        {
            if (!me->readKeyEvent(me))
            {
                LOG_WARN((demo, "Keyboard helper, select failed"));
                me->running = false;
            }
            else
            {
                LOGD_DEBUG((demo, "keyboard removed"));
            }
        }
        else if (ret == 0)
        {
        }
        else
        {
            LOG_WARN((demo, "Keyboard helper: %d (%s)", errno, strerror(errno)));
        }
    } // while

    return nullptr;
}

bool KeyboardHelper::selectKeyboard(std::string& outInterface, std::string& outName)
{
    const int maxCount = 10; // max input devices to try

    char name[256];
    int count = 0 ;

    for (count = 0; count < maxCount; count++)
    {
        outInterface = "/dev/input/event" + std::to_string(count);
        fd = open(outInterface.c_str(), O_RDONLY);
        if (fd > 0)
        {
            ioctl(fd, EVIOCGNAME(sizeof(name)), name);
            close(fd);

            outName = name;
            // convert to lower case
            std::transform(outName.begin(), outName.end(), outName.begin(), ::tolower);

            if (outName.find("keyboard") != std::string::npos)
            {
                LOGD_DEBUG((demo, "keyboard \"%s\" found at %s", outName.c_str(),
                        outInterface.c_str()));
                return true;
            }
        }
    }

    LOG_WARN((demo, "no keyboard event interface found"));

    return false;
}

bool KeyboardHelper::readKeyEvent(KeyboardHelper* reader)
{
    const int eventBuffSize = 64;

    int rd;
    bool readKeyEventStatus;
    std::string uuidString;
    std::string uuid;
    struct input_event eventBuffer[eventBuffSize];

    readKeyEventStatus = true;
    rd = read(reader->fd, eventBuffer, sizeof(struct input_event) * eventBuffSize);
    if (isKeyPressEvent(eventBuffer))
    {
        receiver->OnKey(eventBuffer[1].code, (bool)eventBuffer[1].value);
    }

    if (rd == -1) // device removed
    {
        LOGD_DEBUG((demo, "Keyboard helper, keyboard removed"));
        close(reader->fd);
        reader->fd = 0;
        readKeyEventStatus = false;
    }
    else if (rd >= 0)
    {
        LOGD_DEBUG((demo, "Keyboard helper, rd = %d", rd));
    }

    return readKeyEventStatus;
}

bool KeyboardHelper::isKeyPressEvent(struct input_event Event[])
{
    bool status = true;

    if ((Event[1].value == 1) && (Event[1].type == EV_KEY))
    {
        LOGD_VERBOSE((demo, "key %d pressed", Event[1].code));
    }
    else if ((Event[1].value == 0) && (Event[1].code != 0))
    {
        LOGD_VERBOSE((demo, "key %d released", Event[1].code));
    }
    else
    {
        /* Continuous key press */
        status = false;
    }
    return status;
}

} } // namespace adit { namespace aauto
